/**
 * \file: mspin_connection_tcp_manager.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * mySPIN TLS/TCP Connection Manager
 *
 * \component: MSPIN
 *
 * \author: Thilo Bjoern Fickel BSOT/PJ-ES1 thilo.fickel@bosch-softtec.com
 *
* \copyright: (c) 2003 - 2016 ADIT Corporation
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#include "mspin_connection_tcp_manager.h"
#include "mspin_logging.h"
#include "mspin_connection_tcp_listener.h"

#include <openssl/err.h>
#include <openssl/x509v3.h> //X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS
#include <arpa/inet.h>      //inet_ntoa
#include <sys/prctl.h>      //prctl
#include <net/if_arp.h>     //arpreq, ARPHRD_ETHER
#include <sys/ioctl.h>      //ioctl, SIOCGARP
//#include <sys/socket.h>   //getsockname
#include <errno.h>          //errno
#include <ifaddrs.h>        //getifaddrs, freeifaddrs, ifaddrs

#define MSPIN_TCP_SERVER_MAX_SOCKETS 10
#define MSPIN_TCP_DATA_MESSAGE_BUFFER_SIZE 20
#define MSPIN_TCP_WATCH_THREAD_TIMEOUT_MS 5000
#define MSPIN_TCP_CONTROL_MESSAGE_BUFFER_SIZE 1

typedef struct
{
    int pipeFD[2];
    pthread_t watchThreadID;
    BOOL watchThreadQuit;
} mspin_tcp_watchContext_t;

mspin_tcp_watchContext_t* gpTCPWatchContext = NULL;
mspin_tcp_connection_t* gTCPConnections[MSPIN_TCP_SERVER_MAX_SOCKETS] = {0};

static int mspin_tcp_unlockConnection(const int connectionIndex)
{

    mspin_log_printLn(eMspinVerbosityVerbose, "%s(index=%d) unlock",__FUNCTION__, connectionIndex);
    /* PRQA: Lint Message 455: The thread mutex is unclocked after being locked. So ignore this error */
    /*lint -e455*/
    return pthread_mutex_unlock(&(gTCPConnections[connectionIndex]->mutex));
    /*lint +e455*/
}

static int mspin_tcp_lockConnection(const int connectionIndex)
{
    if ((connectionIndex < 0) || (connectionIndex >= MSPIN_TCP_SERVER_MAX_SOCKETS))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(index=%d) ERROR: Connection index is out of bounds (0 - %d)",
                __FUNCTION__, connectionIndex, MSPIN_TCP_SERVER_MAX_SOCKETS);
        return -2;
    }

    if (!gTCPConnections[connectionIndex])
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(index=%d) ERROR: Connection is NULL",
                __FUNCTION__, connectionIndex);
        return -1;
    }

    mspin_log_printLn(eMspinVerbosityVerbose, "%s(index=%d) lock",__FUNCTION__, connectionIndex);
    /*PRQA: Lint Message 454: The threat mutex will be unlocked after being locked. So ignore this error */
    /*lint -e454*/
    return pthread_mutex_lock(&(gTCPConnections[connectionIndex]->mutex));
    /*lint +e454*/
}

static int mspin_tcp_trylockConnection(const int connectionIndex)
{
    if ((connectionIndex < 0) || (connectionIndex >= MSPIN_TCP_SERVER_MAX_SOCKETS))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(index=%d) ERROR: Connection index is out of bounds (0 - %d)",
                __FUNCTION__, connectionIndex, MSPIN_TCP_SERVER_MAX_SOCKETS);
        return -2;
    }

    if (!gTCPConnections[connectionIndex])
    {
        return -1;
    }

    mspin_log_printLn(eMspinVerbosityVerbose, "%s(index=%d) trylock",__FUNCTION__, connectionIndex);
    return pthread_mutex_trylock(&(gTCPConnections[connectionIndex]->mutex));
}

static int mspin_tcp_findConnection(const int socketFD)
{
    unsigned int i = 0;
    for (i = 0; i < MSPIN_TCP_SERVER_MAX_SOCKETS; i++)
    {
        if (gTCPConnections[i] && (gTCPConnections[i]->socketFD == socketFD))
        {
            return i;
        }
    }

    return -1;
}

static int mspin_tcp_findAndLockConnection(const int socketFD)
{
    unsigned int i = 0;
    for (i = 0; i < MSPIN_TCP_SERVER_MAX_SOCKETS; i++)
    {
        if (gTCPConnections[i] && (gTCPConnections[i]->socketFD == socketFD))
        {
            (void)mspin_tcp_lockConnection(i);
            return i;
        }
    }

    return -1;
}

//Close socket w/o retrieving lock. This has to be done outside
static void mspin_tcp_closeConnectionByIndex(const int connectionIndex)
{
    if ((connectionIndex < 0) || (connectionIndex >= MSPIN_TCP_SERVER_MAX_SOCKETS))
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(index=%d) FATAL ERROR: Connection index is out of bounds (0 - %d)",
                __FUNCTION__, connectionIndex, MSPIN_TCP_SERVER_MAX_SOCKETS);
        return;
    }

    if (!gTCPConnections[connectionIndex])
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(index=%d) FATAL ERROR: connectionIndex invalid",
                __FUNCTION__, connectionIndex);
        return;
    }

    close(gTCPConnections[connectionIndex]->socketFD);
    gTCPConnections[connectionIndex]->socketFD = -1;
    gTCPConnections[connectionIndex]->state = MSPIN_CONNECTION_CLOSED;
    gTCPConnections[connectionIndex]->ipAddr = 0;

    //Release SSL structure
    if (gTCPConnections[connectionIndex]->pSSL)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(index=%d) Calling SSL_free with SSL=%p",
             __FUNCTION__, connectionIndex, gTCPConnections[connectionIndex]->pSSL);

        SSL_free(gTCPConnections[connectionIndex]->pSSL);
        gTCPConnections[connectionIndex]->pSSL = NULL;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityVerbose, "%s(index=%d) pSSL is NULL", __FUNCTION__, connectionIndex);
    }
}

const char* mspin_tcp_getReasonName(MSPIN_TCP_CONNECTION_END_REASON reason)
{
    switch (reason)
    {
        case MSPIN_TCP_CONNECTION_REJECTED_BY_USER:
        {
            return "Rejected by user";
            //break;
        }
        case MSPIN_TCP_CONNECTION_MAX_CONNECTIONS_REACHED:
        {
            return "Max number of connections reached";
            //break;
        }
        case MSPIN_TCP_CONNECTION_MYSPIN_SESSION_ENDED:
        {
            return "mySPIN session ended";
            //break;
        }
        case MSPIN_TCP_CONNECTION_REMOTE_END_HANG_UP:
        {
            return "Remote end hang up";
            //break;
        }
        case MSPIN_TCP_CONNECTION_SOCKET_ERROR:
        {
            return "Communication error";
            //break;
        }
        case MSPIN_TCP_CONNECTION_OTHER_ERROR:
        {
            return "Other error";
            //break;
        }
        default:
        {
            return "UNKNOWN";
            //break;
        }
    }
}

static void mspin_tcp_getNetworkInterface(const int fd, char *interface, size_t len)
{
    struct sockaddr_in addr = {0};
    struct ifaddrs* ifaddr = NULL;
    struct ifaddrs* ifa = NULL;
    socklen_t addr_len = 0;
    bool found = FALSE;

    if (!interface)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d, interface=%p, len=%u) ERROR: Interface is NULL ",
                __FUNCTION__, fd, interface, (unsigned int)len);
        return;
    }

    addr_len = sizeof (addr);

    /* PRQA: Lint Message 64: The type cast is correct. So ignore this error */
    /*lint -save -e64*/
    if (0 != getsockname(fd, (struct sockaddr*)&addr, &addr_len))
    /*lint -restore*/
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d, interface=%p, len=%u) ERROR: getsockname failed with '%s'(%d)",
                __FUNCTION__, fd, interface, (unsigned int)len, strerror(errno), errno);
        return;
    }

    if (0 != getifaddrs(&ifaddr))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d, interface=%p, len=%u) ERROR: getifaddrs failed with '%s'(%d)",
                __FUNCTION__, fd, interface, (unsigned int)len, strerror(errno), errno);
        return;
    }

    //Check which interface contains the wanted IP
    //When found, ifa->ifa_name contains the name of the interface (eth0, eth0, ppp0,...)
    for (ifa = ifaddr; ifa != NULL; ifa = ifa->ifa_next)
    {
        if (AF_INET == ifa->ifa_addr->sa_family)
        {
            struct sockaddr_in *inaddr = (struct sockaddr_in*)ifa->ifa_addr;
            if (inaddr && (inaddr->sin_addr.s_addr == addr.sin_addr.s_addr))
            {
                if (ifa->ifa_name)
                {
                    //Found it
                    strncpy(interface, ifa->ifa_name, len); //Terminating '\0' might not be copied but unlikely
                    found = TRUE;

                    mspin_log_printLn(eMspinVerbosityDebug, "%s(fd=%d, interface=%p, len=%u) using interface '%s'",
                            __FUNCTION__, fd, interface, (unsigned int)len, interface ? interface : "null");

                    break;
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d, interface=%p, len=%u) ERROR: ifa->ifa_name is NULL",
                            __FUNCTION__, fd, interface, (unsigned int)len);
                }
            }
        }
    }

    if (!found)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d, interface=%p, len=%u) ERROR: Interface not found",
                __FUNCTION__, fd, interface, (unsigned int)len);
    }

    freeifaddrs(ifaddr);
}

static void mspin_tcp_macAddrToAscii(struct sockaddr *addr, char* macAddrString, unsigned int length)
{
    unsigned char *ptr = (unsigned char *)addr->sa_data;

    snprintf(macAddrString, length, "%02X:%02X:%02X:%02X:%02X:%02X",
        (ptr[0] & 0xff), (ptr[1] & 0xff), (ptr[2] & 0xff),
        (ptr[3] & 0xff), (ptr[4] & 0xff), (ptr[5] & 0xff));
}

static void mspin_tcp_getMacAddressFromARPCache(const int fd, const in_addr_t ipAddr, const char* interface, char* macAddr, unsigned int length)
{
    //Get MAC address from arp cache
    struct arpreq arpr = {0};
    struct sockaddr_in *sin = NULL;
    char hostString[INET_ADDRSTRLEN] = "";

    inet_ntop(AF_INET, &(ipAddr), hostString, INET_ADDRSTRLEN);

    if (!interface)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d, ipAddr=%s, interface='%s', length=%u) ERROR: Interface is NULL",
                __FUNCTION__, fd, hostString, interface ? interface : "n/a", length);
        return;
    }

    memset(&arpr, 0, sizeof(arpr));
    sin = (struct sockaddr_in *) &arpr.arp_pa;
    sin->sin_family = AF_INET;
    sin->sin_addr.s_addr = ipAddr;
    sin = (struct sockaddr_in *) &arpr.arp_ha;
    sin->sin_family = ARPHRD_ETHER;

    strncpy(arpr.arp_dev, interface, 15); //copy at max 15 bytes as arp_dev is 16 bytes

    if (-1 == ioctl(fd, SIOCGARP, &arpr))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d, ipAddr=%s, interface='%s', length=%u) ERROR: Getting MAC address using ioctl failed with '%s'(=%d)",
                __FUNCTION__, fd, hostString, interface ? interface : "null", length, strerror(errno), errno);
    }
    else
    {
        mspin_tcp_macAddrToAscii(&arpr.arp_ha, macAddr, length);

        mspin_log_printLn(eMspinVerbosityDebug, "%s(fd=%d, ipAddr=%s, interface='%s', length=%u) macAddr='%s'",
                __FUNCTION__, fd, hostString, interface ? interface : "null", length, macAddr);
    }
}

//Find an empty connection. If found create a new connection and lock it immediately
MSPIN_ERROR mspin_tcp_createNewConnection(const int socketFD, const in_addr_t ipAddr)
{
    unsigned int i = 0;
    int rc = 0;
    char hostString[INET_ADDRSTRLEN] = "";
    char networkInterface[15] = "";

    inet_ntop(AF_INET, &(ipAddr), hostString, INET_ADDRSTRLEN);

    for (i = 0; i < MSPIN_TCP_SERVER_MAX_SOCKETS; i++)
    {
        if (NULL == gTCPConnections[i])
        {
            //Connection is not allocated => allocate a new one
            gTCPConnections[i] = malloc(sizeof(mspin_tcp_connection_t));
            memset(gTCPConnections[i], 0, sizeof(mspin_tcp_connection_t));

            rc = pthread_mutex_init(&(gTCPConnections[i]->mutex), NULL);
            if (0 != rc)
            {
                mspin_log_printLn(eMspinVerbosityFatal,
                        "%s(fd=%d, ipAddr=%s) FATAL ERROR: Initializing mutex failed with %d",
                        __FUNCTION__, socketFD, hostString, rc);
                return MSPIN_ERROR_GENERAL;
            }

            rc = mspin_tcp_lockConnection(i);
            if (0 != rc)
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d, ipAddr=%s) ERROR: Acquiring lock failed with %d",
                        __FUNCTION__, socketFD, hostString, rc);

                return MSPIN_ERROR_GENERAL;
            }

            mspin_log_printLn(eMspinVerbosityDebug,
                    "%s(fd=%d, ipAddr=%s) New connection=%p allocated using connectionIndex=%d",
                    __FUNCTION__, socketFD, hostString, gTCPConnections[i], i);

            //Assign start values
            gTCPConnections[i]->socketFD = socketFD;
            gTCPConnections[i]->ipAddr = ipAddr;
            gTCPConnections[i]->state = MSPIN_CONNECTION_HANDSHAKE;
            gTCPConnections[i]->error = FALSE;

#ifdef MSPIN_TCP_USE_MESSAGE_QUEUE
            gTCPConnections[i]->mesgQueue = -1;
#endif //#ifdef MSPIN_TCP_USE_MESSAGE_QUEUE

            //Get MAC address
            mspin_tcp_getNetworkInterface(socketFD, networkInterface, sizeof(networkInterface));
            mspin_tcp_getMacAddressFromARPCache(socketFD, ipAddr, networkInterface, gTCPConnections[i]->macAddr, sizeof(gTCPConnections[i]->macAddr));

            mspin_tcp_unlockConnection(i);

            return MSPIN_SUCCESS; //leave loop
        }
        else
        {
            if (0 == mspin_tcp_trylockConnection(i))
            {
                if (MSPIN_CONNECTION_CLOSED == gTCPConnections[i]->state)
                {
                    mspin_log_printLn(eMspinVerbosityDebug,
                            "%s(fd=%d, ipAddr=%s) Connection=%p found in close state with connectionIndex=%d -> use",
                            __FUNCTION__, socketFD, hostString, gTCPConnections[i], i);

                    //Assign start values
                    gTCPConnections[i]->socketFD = socketFD;
                    gTCPConnections[i]->ipAddr = ipAddr;
                    gTCPConnections[i]->state = MSPIN_CONNECTION_HANDSHAKE;
                    gTCPConnections[i]->error = FALSE;

#ifdef MSPIN_TCP_USE_MESSAGE_QUEUE
                    gTCPConnections[i]->mesgQueue = -1;
#endif //#ifdef MSPIN_TCP_USE_MESSAGE_QUEUE

                    //Get MAC address
                    mspin_tcp_getNetworkInterface(socketFD, networkInterface, sizeof(networkInterface));
                    mspin_tcp_getMacAddressFromARPCache(socketFD, ipAddr, networkInterface, gTCPConnections[i]->macAddr, sizeof(gTCPConnections[i]->macAddr));

                    if (gTCPConnections[i]->pSSL)
                    {
                        mspin_log_printLn(eMspinVerbosityWarn,
                                "%s(fd=%d, ipAddr=%s) WARNING: Connection found with pointer to SSL=%p -> free",
                                __FUNCTION__, socketFD, hostString, gTCPConnections[i]->pSSL);

                        SSL_free(gTCPConnections[i]->pSSL);
                        gTCPConnections[i]->pSSL = NULL;
                    }

                    //Unlock and leave loop and function
                    mspin_tcp_unlockConnection(i);
                    return MSPIN_SUCCESS;
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityDebug,
                            "%s(fd=%d, ipAddr=%s) Connection=%p (connectionIndex=%d) is not in closed state -> find other",
                            __FUNCTION__, socketFD, hostString, gTCPConnections[i], i);
                }

                mspin_tcp_unlockConnection(i);
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityDebug,
                        "%s(fd=%d, ipAddr=%s) Could not get lock of connection=%p (connectionIndex=%d) -> seems to be in use, try next",
                        __FUNCTION__, socketFD, hostString, gTCPConnections[i], i);
            }
        }
    }

    mspin_log_printLn(eMspinVerbosityError,
            "%s(fd=%d, ipAddr=%s) ERROR: No empty connection found! Max number is %d. Release connections first!",
            __FUNCTION__, socketFD, hostString, MSPIN_TCP_SERVER_MAX_SOCKETS);

    return MSPIN_ERROR_NOT_FOUND;
}

in_addr_t mspin_tcp_getIPAddr(const int socketFD)
{
    in_addr_t ipAddr = 0;
    int connectionIndex = mspin_tcp_findAndLockConnection(socketFD);

    if ((connectionIndex > -1))
    {
        ipAddr = gTCPConnections[connectionIndex]->ipAddr;
        mspin_tcp_unlockConnection(connectionIndex);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d) ERROR: Connection not found", __FUNCTION__, socketFD);
    }

    return ipAddr;
}

MSPIN_ERROR mspin_tcp_getStoredMacAddress(const int socketFD, char* macAddress)
{
    int connectionIndex = 0;

    //Check MAC address pointer
    if (!macAddress)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d, macAddress=%p) ERROR: macAddress is NULL",
                __FUNCTION__, socketFD, macAddress);
        return MSPIN_ERROR_NULL_HANDLE;
    }

    connectionIndex = mspin_tcp_findAndLockConnection(socketFD);

    //Retrieve MAC address
    if ((connectionIndex > -1))
    {
        MSPIN_ERROR result = MSPIN_ERROR_GENERAL;
        if (gTCPConnections[connectionIndex] && (strlen(gTCPConnections[connectionIndex]->macAddr) > 0))
        {
            strncpy(macAddress, gTCPConnections[connectionIndex]->macAddr, sizeof(gTCPConnections[connectionIndex]->macAddr)); //the size of macAddr is 18 bytes
            result = MSPIN_SUCCESS;

            mspin_log_printLn(eMspinVerbosityDebug, "%s(fd=%d, macAddress=%p) returning MAC='%s'",
                    __FUNCTION__, socketFD, macAddress, macAddress);
        }
        else if (!gTCPConnections[connectionIndex])
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d, macAddress=%p) ERROR: Connection not found",
                    __FUNCTION__, socketFD, macAddress);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d, macAddress=%p) ERROR: Connection found but no MAC address present",
                    __FUNCTION__, socketFD, macAddress);
        }

        mspin_tcp_unlockConnection(connectionIndex);
        return result;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d, macAddress=%p) ERROR: Connection not found",
                __FUNCTION__, socketFD, macAddress);
        return MSPIN_ERROR_NOT_FOUND;
    }
}

bool mspin_tcp_setActive(const int socketFD)
{
    bool result = FALSE;
    int connectionIndex = mspin_tcp_findAndLockConnection(socketFD);

    if ((connectionIndex > -1))
    {
        gTCPConnections[connectionIndex]->state = MSPIN_CONNECTION_ACTIVE;
        result = TRUE;
        mspin_tcp_unlockConnection(connectionIndex);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d) ERROR: Connection not found", __FUNCTION__, socketFD);
    }

    return result;
}

bool mspin_tcp_setAccepted(const int socketFD)
{
    bool result = FALSE;
    int connectionIndex = mspin_tcp_findAndLockConnection(socketFD);

    if ((connectionIndex > -1))
    {
        if (MSPIN_CONNECTION_ACTIVE == gTCPConnections[connectionIndex]->state)
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(fd=%d) Connection already in active state -> skip update", __FUNCTION__, socketFD);
            result = FALSE;
            mspin_tcp_unlockConnection(connectionIndex);
        }
        else
        {
            gTCPConnections[connectionIndex]->state = MSPIN_CONNECTION_ACCEPTED;
            result = TRUE;
            mspin_tcp_unlockConnection(connectionIndex);
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d) ERROR: Connection not found", __FUNCTION__, socketFD);
    }

    return result;
}

bool mspin_tcp_getError(const int socketFD)
{
    bool result = FALSE;
    int connectionIndex = mspin_tcp_findAndLockConnection(socketFD);
    if (connectionIndex > -1)
    {
        result = gTCPConnections[connectionIndex]->error;
        mspin_tcp_unlockConnection(connectionIndex);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d) ERROR: Connection not found -> return TRUE",
                __FUNCTION__, socketFD);
        result = TRUE; //return error in case the connection was not found
    }

    return result;
}

bool mspin_tcp_setError(const int socketFD, bool error)
{
    bool result = FALSE;
    int connectionIndex = mspin_tcp_findAndLockConnection(socketFD);

    if (connectionIndex > -1)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(fd=%d, error=%s) Set error",
                __FUNCTION__, socketFD, error ? "true" : "false");

        gTCPConnections[connectionIndex]->error = error;
        result = TRUE;
        mspin_tcp_unlockConnection(connectionIndex);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d, error=%s) ERROR: Connection not found",
                __FUNCTION__, socketFD, error ? "true" : "false");
    }

    return result;
}

//Locks the connection. Unlock is required with mspin_tcp_unlockSSL
SSL* mspin_tcp_lockSSL(const int socketFD)
{
    int connectionIndex = mspin_tcp_findAndLockConnection(socketFD);
    if (connectionIndex > -1)
    {
        return gTCPConnections[connectionIndex]->pSSL;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d) ERROR: Connection not found", __FUNCTION__, socketFD);
        return NULL;
    }
}

void mspin_tcp_unlockSSL(const int socketFD)
{
    int connectionIndex = mspin_tcp_findConnection(socketFD);
    if (connectionIndex > -1)
    {
        mspin_tcp_unlockConnection(connectionIndex);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d) ERROR: Connection not found", __FUNCTION__, socketFD);
    }
}

mspin_tls_accept_result_t mspin_tls_acceptConnection(const int socketFD, SSL_CTX* pSSLContext, bool verifyClient)
{
    int rc = -1;
    int connectionIndex = -1;
    mspin_tls_accept_result_t result = MSPIN_TLS_SETUP_ERROR;

    //Check SSL_CTX handle
    if (!pSSLContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d, SSL_CTX=%p) ERROR: SSL_CTX is NULL",
                __FUNCTION__, socketFD, pSSLContext);
        return MSPIN_TLS_SETUP_ERROR;
    }

    connectionIndex = mspin_tcp_findAndLockConnection(socketFD);
    if (connectionIndex < 0)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d, SSL_CTX=%p) ERROR: Connection with this FD not found",
                __FUNCTION__, socketFD, pSSLContext);
        return MSPIN_TLS_CONNECTION_ERROR;
    }

    //Check if pointer is present. If not we have a severe problem
    if (!gTCPConnections[connectionIndex])
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(fd=%d, SSL_CTX=%p) FATAL ERROR: Connection is NULL. This may not happen because connection got locked",
                __FUNCTION__, socketFD, pSSLContext);
        return MSPIN_TLS_CONNECTION_ERROR;
    }

    //Check if connection is in OPEN state
    if (MSPIN_CONNECTION_HANDSHAKE != gTCPConnections[connectionIndex]->state)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d, SSL_CTX=%p) ERROR: Connection not open, state is %d",
                __FUNCTION__, socketFD, pSSLContext, gTCPConnections[connectionIndex]->state);
        mspin_tcp_unlockConnection(connectionIndex);
        return MSPIN_TLS_CONNECTION_ERROR;
    }

    //Create SSL context and assign socket connection to it
    gTCPConnections[connectionIndex]->pSSL = SSL_new(pSSLContext);
    SSL_set_fd(gTCPConnections[connectionIndex]->pSSL, socketFD);

#if OPENSSL_VERSION_NUMBER >= 0x10100000L
    if (pContext->tlsConfig.verifyClient)
    {
//        X509_VERIFY_PARAM *param = SSL_get0_param(pSSL);
//        X509_VERIFY_PARAM_set_hostflags(param, X509_CHECK_FLAG_NO_PARTIAL_WILDCARDS);
//        X509_VERIFY_PARAM_set1_host(param, clientHostname, 0); //ToDo: clientHostname is not known

        mspin_log_printLn(eMspinVerbosityError,
                "%s(fd=%d, SSL_CTX=%p) ERROR: Hostname verification is not yet implemented",
                __FUNCTION__, socketFD, pSSLContext);
    }
#endif //OPENSSL_VERSION_NUMBER >= 0x10100000L

    ERR_clear_error();

    mspin_log_printLn(eMspinVerbosityDebug, "%s(fd=%d, SSL_CTX=%p) before calling SSL_accept",
            __FUNCTION__, socketFD, pSSLContext); //ToDo: Remove or reduce verbosity

    rc = SSL_accept(gTCPConnections[connectionIndex]->pSSL);
    if (rc <= 0)
    {
        if (gTCPConnections[connectionIndex]->pSSL)
        {
            unsigned long sslError = ERR_get_error();

            mspin_log_printLn(eMspinVerbosityError,
                    "%s(fd=%d) ERROR: SSL_accept returned with %d (SSL error code=%d,ERR code='%s'(%lu))",
                    __FUNCTION__, socketFD, rc, SSL_get_error(gTCPConnections[connectionIndex]->pSSL, rc),
                    ERR_error_string(sslError, NULL), sslError);

            //Close connection
            mspin_tcp_closeConnectionByIndex(connectionIndex);

            result = MSPIN_TLS_ACCEPT_ERROR;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d) ERROR: pSSL got NULL during SSL_accept",
                    __FUNCTION__, socketFD);

            //Close connection
            mspin_tcp_closeConnectionByIndex(connectionIndex);
            result = MSPIN_TLS_CONNECTION_ERROR;
        }
    }
    else
    {
        if (verifyClient)
        {
            X509 *peer = SSL_get_peer_certificate(gTCPConnections[connectionIndex]->pSSL);
            if (peer)
            {
                if (SSL_get_verify_result(gTCPConnections[connectionIndex]->pSSL) == X509_V_OK)
                {
                    mspin_log_printLn(eMspinVerbosityInfo, "%s(fd=%d) Client certification succeeded",
                            __FUNCTION__, socketFD);
                    result = MSPIN_TLS_CONNECTED;
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d) ERROR: Client certification failed -> close connection",
                            __FUNCTION__, socketFD);

                    //Close connection
                    mspin_tcp_closeConnectionByIndex(connectionIndex);

                    result = MSPIN_TLS_VERIFICATION_ERROR;
                }
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError,
                        "%s(fd=%d) ERROR: Client did not send a certificate (or sent an empty certificate)",
                        __FUNCTION__, socketFD);

                //Close connection
                mspin_tcp_closeConnectionByIndex(connectionIndex);

                result = MSPIN_TLS_NO_CLIENT_CERT;
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(fd=%d) client not verified", __FUNCTION__, socketFD);

            result = MSPIN_TLS_CONNECTED;
        }
    }

    mspin_tcp_unlockConnection(connectionIndex);
    return result;
}

MSPIN_ERROR mspin_tcp_closeConnection(const int socketFD)
{
    char hostString[INET_ADDRSTRLEN] = "";
    int connectionIndex = -1;

    mspin_log_printLn(eMspinVerbosityDebug, "%s(fd=%d) entered", __FUNCTION__, socketFD);

    connectionIndex = mspin_tcp_findAndLockConnection(socketFD);
    if ((connectionIndex < 0) || (!gTCPConnections[connectionIndex]))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d) ERROR: Connection not found", __FUNCTION__, socketFD);
        return MSPIN_ERROR_NOT_FOUND;
    }

    inet_ntop(AF_INET, &(gTCPConnections[connectionIndex]->ipAddr), hostString, INET_ADDRSTRLEN);

    mspin_tcp_closeConnectionByIndex(connectionIndex);

    mspin_tcp_unlockConnection(connectionIndex); //Only unlock. Do not free the memory of the connection in order to allow reuse

    mspin_log_printLn(eMspinVerbosityInfo, "%s(fd=%d) fd closed for ipAddr='%s'",
            __FUNCTION__, socketFD, hostString);

    return MSPIN_SUCCESS;
}

static void mspin_tcp_createWatchContext(void)
{
    if (!gpTCPWatchContext)
    {
        gpTCPWatchContext = malloc(sizeof(mspin_tcp_watchContext_t));

        if (!gpTCPWatchContext)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to allocate memory for watch context",
                    __FUNCTION__);
            return;
        }

        memset(gpTCPWatchContext, 0, sizeof(mspin_tcp_watchContext_t));

        mspin_log_printLn(eMspinVerbosityDebug, "%s() watch context=%p created", __FUNCTION__, gpTCPWatchContext);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s() WARNING: Watch context is already present -> reuse %p",
                __FUNCTION__, gpTCPWatchContext);
    }
}

static void mspin_tcp_deleteWatchContext(void)
{
    void* status = NULL;

    if (!gpTCPWatchContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: No watch context present", __FUNCTION__);
        return;
    }

    if (gpTCPWatchContext->watchThreadID)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() Watch thread is still present -> join thread first",
                __FUNCTION__);

        //Mark watch thread to terminate
        gpTCPWatchContext->watchThreadQuit = TRUE;

        //Send control message to watch thread to unblock the watch thread and let him leave the loop
        mspin_tcp_signalNewConnection();

        //Close pipe used in watch thread
        close(gpTCPWatchContext->pipeFD[0]);
        gpTCPWatchContext->pipeFD[0] = -1;
        close(gpTCPWatchContext->pipeFD[1]);
        gpTCPWatchContext->pipeFD[1] = -1;

        (void)pthread_join(gpTCPWatchContext->watchThreadID, &status);
        gpTCPWatchContext->watchThreadID = 0;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s() WARNING: No watch thread present", __FUNCTION__);
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s() Deleting watch context=%p now",
            __FUNCTION__, gpTCPWatchContext);

    free(gpTCPWatchContext);
    gpTCPWatchContext = NULL;
}

//Send control message (an empty message with one byte)
static MSPIN_ERROR mspin_tcp_sendControlMsg(int fd)
{
    int writeResult = 0;
    U8 buffer[1] = "\0";

    writeResult = write(fd, buffer, MSPIN_TCP_CONTROL_MESSAGE_BUFFER_SIZE);
    if (writeResult == -1)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(fd=%d) ERROR: write failed with '%s'(%d)",
                __FUNCTION__, fd, strerror(errno), errno);
        return MSPIN_ERROR_GENERAL;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(fd=%d) %d bytes written",
                __FUNCTION__, fd, writeResult);
        return MSPIN_SUCCESS;
    }
}

static void* mspin_tcp_watchThread(void* exinf)
{
    fd_set fds = {{0}};
    struct timeval tv;
    int selectResult = 0;
    int highestFD = 0;
    int i = 0;
    char controlMsg[MSPIN_TCP_CONTROL_MESSAGE_BUFFER_SIZE];
    char dataMsg[MSPIN_TCP_DATA_MESSAGE_BUFFER_SIZE];
    int readResult = 0;
    ssize_t bytes_read = 0;

    //Set thread name
    prctl(PR_SET_NAME, "mspin_tcpWatch", 0, 0, 0);

    mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) entered", __FUNCTION__, exinf);

    while (!gpTCPWatchContext->watchThreadQuit && (gpTCPWatchContext->pipeFD[0] != -1))
    {
        /* PRQA: Lint Message 529: Warning due symbol not subsequently referenced in the asm section. Not an issue */
        /*lint -save -e529*/
        FD_ZERO(&fds);
        /*lint -restore*/

        //Add control file descriptor
        FD_SET(gpTCPWatchContext->pipeFD[0], &fds);
        highestFD = gpTCPWatchContext->pipeFD[0];

        //Add socket FDs
        for (i = 0; i < MSPIN_TCP_SERVER_MAX_SOCKETS; i++)
        {
            if (0 == mspin_tcp_trylockConnection(i)) //checks also gTCPConnections[i]
            {
                if ((gTCPConnections[i]->socketFD != -1) && (gTCPConnections[i]->state == MSPIN_CONNECTION_ACCEPTED))
                {
                    FD_SET(gTCPConnections[i]->socketFD, &fds);
                    if (gTCPConnections[i]->socketFD > highestFD)
                    {
                        highestFD = gTCPConnections[i]->socketFD;
                    }
                }
                else if (gTCPConnections[i]->socketFD == -1)
                {
                    //Socket not in use, nothing to do
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityVerbose, "%s(exinf=%p) socket=%d is in use and will not be observed",
                            __FUNCTION__, exinf, gTCPConnections[i]->socketFD);
                }

                mspin_tcp_unlockConnection(i);
            }
            else if (gTCPConnections[i])
            {
                mspin_log_printLn(eMspinVerbosityWarn, "%s(exinf=%p) WARNING: Failed to get lock for index=%d",
                        __FUNCTION__, exinf, i);
            }
        }

        tv.tv_sec = MSPIN_TCP_WATCH_THREAD_TIMEOUT_MS/1000; //seconds
        tv.tv_usec = (MSPIN_TCP_WATCH_THREAD_TIMEOUT_MS%1000)*1000; //milliseconds

        selectResult = select(highestFD+1, &fds, NULL, NULL, &tv);
        {
            if (gpTCPWatchContext)
            {
                if ((selectResult > 0) && FD_ISSET(gpTCPWatchContext->pipeFD[0], &fds))
                {
                    mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) control msg received",
                            __FUNCTION__, exinf);
                    bytes_read = read(gpTCPWatchContext->pipeFD[0], controlMsg, MSPIN_TCP_CONTROL_MESSAGE_BUFFER_SIZE);

                    mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) bytes read = %zd",
                            __FUNCTION__, exinf, bytes_read);
                }
                else if (selectResult < 0)
                {
                    mspin_log_printLn(eMspinVerbosityError, "%s(exinf=%p) ERROR: select returned error '%s'(%d)",
                            __FUNCTION__, exinf, strerror(errno), errno);

                    gpTCPWatchContext->watchThreadQuit = TRUE;
                }
                else if (selectResult == 0)
                {
                    //Timeout, nothing to do
                    mspin_log_printLn(eMspinVerbosityVerbose, "%s(exinf=%p) timeout", __FUNCTION__, exinf);
                }
                else
                {
                    for (i = 0; i < MSPIN_TCP_SERVER_MAX_SOCKETS; i++)
                    {
                        if (0 == mspin_tcp_trylockConnection(i))
                        {
                            if (FD_ISSET(gTCPConnections[i]->socketFD, &fds))
                            {
                                if (gTCPConnections[i]->state == MSPIN_CONNECTION_ACCEPTED)
                                {
                                    if (gTCPConnections[i]->pSSL)
                                    {
                                        readResult = SSL_read(gTCPConnections[i]->pSSL, (char*)dataMsg, MSPIN_TCP_DATA_MESSAGE_BUFFER_SIZE);
                                        if (readResult > 0)
                                        {
                                            mspin_log_printLn(eMspinVerbosityWarn,
                                                    "%s(exinf=%p) WARNING: %d bytes read on TLS secured fd=%d (Starting with: '%02x %02x %02x %02x')",
                                                    __FUNCTION__, exinf, readResult, gTCPConnections[i]->socketFD,
                                                    dataMsg[0], dataMsg[1], dataMsg[2], dataMsg[3]);
                                        }
                                        else if (0 == readResult)
                                        {
                                            int error = SSL_get_error((const SSL*)gTCPConnections[i]->pSSL, readResult);
                                            unsigned long detailedError = ERR_get_error();

                                            if (SSL_ERROR_ZERO_RETURN == error)
                                            {
                                                mspin_log_printLn(eMspinVerbosityError,
                                                        "%s(exinf=%p) ERROR: TLS connection shut down",
                                                        __FUNCTION__, exinf);

                                                mspin_tcp_signalConnectionClosed(gTCPConnections[i]->socketFD,
                                                        gTCPConnections[i]->ipAddr, MSPIN_TCP_CONNECTION_REMOTE_END_HANG_UP);
                                                mspin_tcp_closeConnectionByIndex(i); //this function does not lock the mutex again
                                            }
                                            else
                                            {
                                                mspin_log_printLn(eMspinVerbosityError,
                                                        "%s(exinf=%p) ERROR: SSL_read failed with rc=%d and error=%d",
                                                        __FUNCTION__, exinf, readResult, error);
                                                mspin_log_printLn(eMspinVerbosityError,
                                                        "%s(exinf=%p) ERROR: Details are '%s'",
                                                        __FUNCTION__, exinf, readResult,
                                                        ERR_error_string(detailedError, NULL));

                                                mspin_tcp_signalConnectionClosed(gTCPConnections[i]->socketFD,
                                                        gTCPConnections[i]->ipAddr, MSPIN_TCP_CONNECTION_TLS_ERROR);
                                                mspin_tcp_closeConnectionByIndex(i); //this function does not lock the mutex again
                                            }
                                        }
                                        else
                                        {
                                            int error = SSL_get_error((const SSL*)gTCPConnections[i]->pSSL, readResult);
                                            unsigned long detailedError = ERR_get_error();

                                            mspin_log_printLn(eMspinVerbosityError,
                                                    "%s(exinf=%p) ERROR: SSL_read failed with rc=%d and error=%d",
                                                    __FUNCTION__, exinf, readResult, error);
                                            mspin_log_printLn(eMspinVerbosityError,
                                                    "%s(exinf=%p) ERROR: Details are '%s'",
                                                    __FUNCTION__, exinf, readResult,
                                                    ERR_error_string(detailedError, NULL));

                                            mspin_tcp_signalConnectionClosed(gTCPConnections[i]->socketFD,
                                                    gTCPConnections[i]->ipAddr, MSPIN_TCP_CONNECTION_TLS_ERROR);
                                            mspin_tcp_closeConnectionByIndex(i); //this function does not lock the mutex again
                                        }
                                    }
                                    else
                                    {
                                        readResult = read(gTCPConnections[i]->socketFD, dataMsg, MSPIN_TCP_DATA_MESSAGE_BUFFER_SIZE);
                                        if (readResult > 0)
                                        {
                                            mspin_log_printLn(eMspinVerbosityWarn,
                                                    "%s(exinf=%p) WARNING: %d bytes read on TCP/IP fd=%d (Starting with: '%02x %02x %02x %02x')",
                                                    __FUNCTION__, exinf, readResult, gTCPConnections[i]->socketFD,
                                                    dataMsg[0], dataMsg[1], dataMsg[2], dataMsg[3]);
                                        }
                                        else if (0 == readResult) //indicates that the socket got closed by remote end
                                        {
                                            mspin_log_printLn(eMspinVerbosityError,
                                                    "%s(exinf=%p) ERROR: 0 read which indicates that remote end closed the socket fd=%d => close socket",
                                                    __FUNCTION__, exinf, gTCPConnections[i]->socketFD);

                                            mspin_tcp_signalConnectionClosed(gTCPConnections[i]->socketFD,
                                                    gTCPConnections[i]->ipAddr, MSPIN_TCP_CONNECTION_REMOTE_END_HANG_UP);
                                            mspin_tcp_closeConnectionByIndex(i); //this function does not lock the mutex again
                                        }
                                        else //error
                                        {
                                            mspin_log_printLn(eMspinVerbosityError,
                                                    "%s(exinf=%p) ERROR: read returned error '%s'(%d) => close socket",
                                                    __FUNCTION__, exinf, strerror(errno), errno);

                                            mspin_tcp_signalConnectionClosed(gTCPConnections[i]->socketFD,
                                                    gTCPConnections[i]->ipAddr, MSPIN_TCP_CONNECTION_SOCKET_ERROR);
                                            mspin_tcp_closeConnectionByIndex(i); //this function does not lock the mutex again
                                        }
                                    }
                                }
                                else if (gTCPConnections[i]->state == MSPIN_CONNECTION_ACTIVE)
                                {
                                    mspin_log_printLn(eMspinVerbosityDebug,
                                            "%s(exinf=%p) socket %d is now in use for mySPIN connection (state=%d)",
                                            __FUNCTION__, exinf, gTCPConnections[i]->socketFD,
                                            gTCPConnections[i]->state);
                                }
                                else
                                {
                                    mspin_log_printLn(eMspinVerbosityError,
                                            "%s(exinf=%p) ERROR: socket %d closed (state=%d)",
                                            __FUNCTION__, exinf, gTCPConnections[i]->socketFD,
                                            gTCPConnections[i]->state);
                                }
                            }
                            mspin_tcp_unlockConnection(i);
                        }
                        else if (gTCPConnections[i])
                        {
                            mspin_log_printLn(eMspinVerbosityWarn,
                                    "%s(exinf=%p) WARNING: Failed to get lock for index=%d", __FUNCTION__, exinf, i);
                        }
                    }
                }
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(exinf=%p) ERROR: Watch context is NULL",
                        __FUNCTION__, exinf);
            }
        }
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) terminate thread", __FUNCTION__, exinf);

    pthread_exit(exinf);
    return NULL;
}

static S32 mspin_tcp_startWatchThread(void)
{
    pthread_attr_t attr;
    S32 rc = -1;

    if (!gpTCPWatchContext)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Watch context is NULL", __FUNCTION__);
        return rc;
    }

    memset(&attr, 0, sizeof(pthread_attr_t));

    rc = pthread_attr_init(&attr);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to initialize thread attributes", __FUNCTION__);
        return rc;
    }

    rc = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to set detach state", __FUNCTION__);
        return false;
    }

    gpTCPWatchContext->watchThreadQuit = FALSE;

    rc = pthread_create(&(gpTCPWatchContext->watchThreadID), &attr, mspin_tcp_watchThread, NULL);
    if (0 == rc)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() using thread id=%d", __FUNCTION__, gpTCPWatchContext->watchThreadID);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to create thread", __FUNCTION__);
        gpTCPWatchContext->watchThreadID = 0;
    }

    (void)pthread_attr_destroy(&attr);

    return rc;
}

MSPIN_ERROR mspin_tcp_startWatching(void)
{
    mspin_tcp_createWatchContext();

    if (gpTCPWatchContext)
    {
        if (-1 == pipe(gpTCPWatchContext->pipeFD))
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Creating pipe failed with '%s'(%d)",
                    __FUNCTION__, strerror(errno), errno);
            return MSPIN_ERROR_GENERAL;
        }

        if (0 != mspin_tcp_startWatchThread())
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to start watch thread", __FUNCTION__);
            return MSPIN_ERROR_GENERAL;
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to allocate watch context", __FUNCTION__);
        return MSPIN_ERROR_GENERAL;
    }

    return MSPIN_SUCCESS;
}

MSPIN_ERROR mspin_tcp_stopWatching(void)
{
    mspin_tcp_deleteWatchContext();
    return MSPIN_SUCCESS;
}

void mspin_tcp_signalNewConnection(void)
{
    mspin_tcp_sendControlMsg(gpTCPWatchContext->pipeFD[1]);
}
